/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * ALSA support by Paul Davis
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>

#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>

#include "driver.h"

#ifdef DRIVER_ALSA

#include <alsa/asoundlib.h>

#define ALSA_PCM_NAME "sweep"

static sw_handle *
open_alsa_pcm (int flags)
{
  int err;
  char *alsa_pcm_name;
  snd_pcm_t * pcm_handle;
  sw_handle * handle;

  printf ("opening alsa pcm\n");

  if ((alsa_pcm_name = getenv ("SWEEP_ALSA_PCM")) == 0) {
    alsa_pcm_name = ALSA_PCM_NAME;
  }

  if ((err = snd_pcm_open(&pcm_handle, alsa_pcm_name,
			  SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK)) < 0) {
    fprintf (stderr, "sweep: unable to open ALSA device %s (%s)\n",
	     alsa_pcm_name, snd_strerror (err));
    return NULL;
  }

  handle = g_malloc0 (sizeof (sw_handle));
  handle->driver_fd = snd_pcm_poll_descriptor (pcm_handle);
  handle->custom_data = pcm_handle;

  return handle;
}

static void
setup_alsa_pcm (sw_handle * handle, sw_format * format)
{
  snd_pcm_t * pcm_handle = (snd_pcm_t *)handle->custom_data;
  snd_pcm_params_t params;
  snd_pcm_params_info_t params_info;
  int err = 0;

  memset (&params, 0, sizeof(params));
  memset (&params_info, 0, sizeof(params_info));

  if (snd_pcm_params_info (pcm_handle, &params_info) < 0) {
    fprintf(stderr, "cannot get audio interface parameters (%s)\n",
	    snd_strerror(err));
    return;
  }

  if (params_info.formats & SND_PCM_FMT_S16_LE) {
    params.format.sfmt = SND_PCM_SFMT_S16_LE;
  } else {
    fprintf (stderr, "audio interface does not support "
	     "linear 16 bit little endian samples\n");
    return;
  }
  
  switch (format->rate) {
  case 44100:
    if (params_info.rates & SND_PCM_RATE_44100) {
      params.format.rate = 44100;
    } else {
      fprintf (stderr, "audio interface does not support "
	       "44.1kHz sample rate (0x%x)\n",
	       params_info.rates);
      return;
    }
    break;
    
  case 48000:
    if (params_info.rates & SND_PCM_RATE_48000) {
      params.format.rate = 48000;
    } else {
      fprintf (stderr, "audio interface does not support "
	       "48kHz sample rate\n");
      return;
    }
    break;
    
  default:
    fprintf (stderr, "audio interface does not support "
	     "a sample rate of %d\n",
	     format->rate);
    return;
  }

  if (format->channels < params_info.min_channels ||
      format->channels > params_info.max_channels) {
    fprintf (stderr, "audio interface does not support %d channels\n",
             format->channels);
    return;
  }

  params.format.channels = format->channels;
  params.ready_mode = SND_PCM_READY_FRAGMENT;
  params.start_mode = SND_PCM_START_DATA;
  params.xrun_mode = SND_PCM_XRUN_FRAGMENT;
  params.frag_size = PBUF_SIZE / params.format.channels;
  params.avail_min = params.frag_size;
  // params.buffer_size = 3 * params.frag_size;

  if ((err = snd_pcm_params (pcm_handle, &params)) < 0) {
    fprintf (stderr, "audio interface could not be configured "
	     "with the specified parameters\n");
    return;
  }

  handle->driver_channels = params.format.channels;

  if (snd_pcm_prepare (pcm_handle) < 0) {
    fprintf (stderr, "audio interface could not be prepared "
	     "for playback\n");
    return;
  }
}

static ssize_t
write_alsa_pcm (sw_handle * handle, void * buf, size_t count)
{
  snd_pcm_t pcm_handle = (snd_pcm_t)handle->custom_data;

  return snd_pcm_write (pcm_handle, buf, count / (2*handle->driver_channels));
}

static void
reset_alsa_pcm (sw_handle * handle)
{
}

static void
flush_alsa_pcm (sw_handle * handle)
{
}

static void
drain_alsa_pcm (sw_handle * handle)
{
  snd_pcm_t pcm_handle = (snd_pcm_t)handle->custom_data;

  if (snd_pcm_drop (pcm_handle) < 0) {
        fprintf (stderr, "audio interface could not be stopped\n");
        return;
  }
  if (snd_pcm_prepare (pcm_handle) < 0) {
        fprintf (stderr, "audio interface could not be re-prepared\n");
        return;
  }
}

static void
close_alsa_pcm (sw_handle * handle)
{
  snd_pcm_t pcm_handle = (snd_pcm_t)handle->custom_data;

  snd_pcm_close (pcm_handle);
}

static sw_driver _driver_alsa = {
  NULL, /* config */
  open_alsa_pcm,
  setup_alsa_pcm,
  NULL,
  write_alsa_pcm,
  reset_alsa_pcm,
  flush_alsa_pcm,
  drain_alsa_pcm,
  close_alsa_pcm,
};

#else

static sw_driver _driver_alsa = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

#endif

sw_driver * driver_alsa = &_driver_alsa;
