/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/* Define this to record all output to files in /tmp */
/* #define RECORD_DEMO_FILES */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>

#ifdef RECORD_DEMO_FILES
#include <sndfile.h>

#if !defined (SNDFILE_1)
#error Recording demo files requires libsndfile version 1
#endif

#endif

#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>
#include <sweep/sweep_typeconvert.h>

#include "play.h"
#include "head.h"
#include "driver.h"

#include "sample-display.h"

#define SCRUB_SLACKNESS 2.0

GMutex * play_mutex;

static sw_handle * handle = NULL;

static GList * active_play_heads = NULL;

/*static int realoffset = 0;*/
static sw_sample * prev_sample = NULL;
static pthread_t player_thread = -1;

static gboolean stop_all = FALSE;

/*
 * update_playmarker ()
 *
 * Update the position of the playback marker line for the sample
 * being played.
 *
 * Called within the main sweep interface thread.
 * gtk_idle will keep calling this function as long as this sample is
 * playing, unless otherwise stopped.
 */
static gint
update_playmarker (gpointer data)
{
  sw_sample * s = (sw_sample *)data;
  sw_head * head = s->play_head;

  if (!sample_bank_contains (s)) {
    return FALSE;
  } else if (!head->going) {

#ifdef DEBUG
    g_print ("update_playmarker: play_mode is STOP\n");
#endif
    s->playmarker_tag = 0;

    /* Set user offset to correct offset */
    if (head->previewing) {
      sample_set_playmarker (s, head->stop_offset, TRUE);
      head_set_previewing (head, FALSE);
    } else {
      sample_set_playmarker (s, (sw_framecount_t)head->offset, TRUE);
    }
    
    /* As this may have been stopped by the player thread, refresh the
     * interface */
    head_set_going (head, FALSE);
    sample_refresh_playmode (s);
    
    return FALSE;
  } else {
    sample_set_playmarker (s, head->realoffset, FALSE);

    return TRUE;
  }
}

static void
start_playmarker (sw_sample * s)
{
  if (s->playmarker_tag > 0) {
    gtk_timeout_remove (s->playmarker_tag);
  }

  s->playmarker_tag =
    gtk_timeout_add ((guint32)30,
		     (GtkFunction)update_playmarker,
		     (gpointer)s);
}

static struct timeval tv_instant = {0, 0};

#define FORWARDS (1)
#define BACKWARDS (0)

#ifdef DEVEL_CODE
/* returns TRUE if it plays all the way to end */
static gboolean
play_view(sw_view * view, sw_framecount_t start, sw_framecount_t end,
	  sw_framecount_t lbound, sw_framecount_t ubound,
	  gfloat relpitch)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;
  sw_format * format;
  fd_set fds;
  ssize_t n = 0;
  sw_audio_t * d;
  gint sbytes, channels;
  gdouble po = 0.0, p, startf, endf;
  gdouble momentum_delta = 0.0;
  gboolean do_smoothing = FALSE;
  gint i=0, si=0;
  gint direction = FORWARDS;
  sw_framecount_t last_user_offset = -1;
  sw_audio_t pbuf[PBUF_SIZE];

  if (handle == NULL) return FALSE;

  d = (sw_audio_t *)s->sounddata->data;

  sbytes = 2;
 
  format = s->sounddata->format;
  channels = format->channels;

  /* compensate for drivers unable to do required sampling rate */
  relpitch *= (format->rate / handle->driver_rate);
  
  startf = (gdouble)(lbound);
  endf = (gdouble)(ubound-channels); /* linear interp wants to look ahead
				      * one frame */

  if (head->reverse) {
    head->offset = (gdouble)end;
  } else {
    head->offset = (gdouble)start;
  }

  po = (gdouble)head->offset;

  memset (pbuf, 0, sizeof (pbuf));

  while ((head->reverse ? (po >= startf) : (po < endf)) &&
	 sample_bank_contains (s) && head->going) {

    FD_ZERO (&fds);
    FD_SET (handle->driver_fd, &fds);
    
    if (select (handle->driver_fd+1, NULL, &fds, NULL, &tv_instant) == 0);

    /*    memset (pbuf, 0, sizeof (pbuf));*/

    for (i = 0; i < PBUF_SIZE; i++) {

      if (head->mute || s->user_offset == last_user_offset ||
	  po < lbound || po >= ubound) {
	pbuf[i] = 0;
	if (channels == 1 && handle->driver_channels == 2) {
	  pbuf[++i] = 0;
	}

      } else {

	switch (channels) {
	case 1:
	  switch (handle->driver_channels) {
	  case 1:
	    /* Mono to mono */
	    si = (int)floor(po);
	    p = po - (gdouble)si;
	    pbuf[i] = head->gain * (d[si] * p + d[si+channels] * (1 - p));
	    break;
	  case 2:
	    /* Mix mono data up to stereo device */
	    si = (int)floor(po);
	    p = po - (gdouble)si;
	    pbuf[i] = head->gain *
	      (d[si] * p + d[si+channels] * (1 - p));
	    pbuf[i+1] = pbuf[i];
	    i++;
	    break;
	  default:
	    break;
	  }
	  break;
	default:
	  switch (handle->driver_channels) {
	  case 1:
	    /* Mix multichannel data down to mono device */
	    si = (int)floor(po);
	    p = po - (gdouble)si;
	    si *= channels;
	    pbuf[i] = head->gain * 0.5 * ((d[si] + d[si+1]) * p +
	       (d[si+channels] + d[si+1+channels]) * (1 - p));
	    break;
	  case 2:
	    /* Stereo/Multichannel to stereo */
	    si = (int)floor(po);
	    p = po - (gdouble)si;
	    si *= channels;
	    pbuf[i] = head->gain * (d[si] * p + d[si+channels] * (1 - p));
	    i++; si++;
	    pbuf[i] = head->gain * (d[si] * p + d[si+channels] * (1 - p));
	    break;
	  default:
	    break;
	  }
      
	  break;
	}

#if 1
	/* If this data came about from scrubbing, it is probably a bit
	 * noisy; run it through a simple smoothing filter:
	 *
	 *   x[n] = ( x[n] + 2 * x[n-1] + 3 * x[n-2] ) / 6
	 */
	if (do_smoothing) {
	  int iL1, iL2;
	  int iR1, iR2;
	  
	  switch (handle->driver_channels) {
	  case 1:
	    iL1 = (i - 1 + PBUF_SIZE) % PBUF_SIZE;
	    iL2 = (i - 2 + PBUF_SIZE) % PBUF_SIZE;
	    pbuf[i] += pbuf[i] * 2.0;
	    pbuf[i] += pbuf[iL1]*3.0 + pbuf[iL2]*4.0;
	    pbuf[i] /= 4.0;
	    break;
	  case 2:
	    iL1 = (i - 2 + PBUF_SIZE) % PBUF_SIZE;
	    iL2 = (i - 4 + PBUF_SIZE) % PBUF_SIZE;
	    iR1 = (i - 3 + PBUF_SIZE) % PBUF_SIZE;
	    iR2 = (i - 5 + PBUF_SIZE) % PBUF_SIZE;
	    pbuf[i] += pbuf[i] * 2.0;
	    pbuf[i] += pbuf[iL1] * 3.0 + pbuf[iL2] * 4.0;
	    pbuf[i] /= 4.0;
	    pbuf[i-1] += pbuf[i-1] * 2.0;
	    pbuf[i-1] += pbuf[iR1] * 3.0 + pbuf[iR2] * 4.0;
	    pbuf[i-1] /= 4.0;
	  }
	  do_smoothing = FALSE;
	}
#else
	
#endif
      }


      if (s->by_user) {
	if (head->scrubbing) {
	  gdouble old_po = po;
	  gdouble play_delta;

	  if (s->user_offset < (int)po) {
	    direction = BACKWARDS;
	  } else {
	    direction = FORWARDS;
	  }

#if 0
	  po = (po + (SCRUB_SLACKNESS-1.0)*(gdouble)s->user_offset) /
	    SCRUB_SLACKNESS;
#elif 1
	  po = (po*(SCRUB_SLACKNESS-1.0) + (gdouble)s->user_offset) /
	    SCRUB_SLACKNESS;
#else
	  po = (gdouble)s->user_offset;
#endif

	  momentum_delta = (po - old_po)/8;
	  momentum_delta = CLAMP(momentum_delta, -10, +10);

#if 0	      
	  play_delta = view->rate * relpitch;
	  play_delta += momentum_delta;
#else
	  play_delta = momentum_delta;
#endif

	  if (direction == FORWARDS && !head->reverse)
	    po += play_delta;
	  else if (direction == BACKWARDS && head->reverse)
	    po -= play_delta;

	  head->offset = (int)po;

	} else {
	  head->offset = s->user_offset;
	  po = (gdouble)head->offset;
	  direction = FORWARDS;
	  momentum_delta = 0;
	}

	last_user_offset = s->user_offset;
	s->by_user = FALSE;

	do_smoothing = TRUE;
      } else {
	gdouble play_delta;

	play_delta = s->rate * relpitch;

	if (head->reverse) {
	  po -= play_delta;
	} else {
	  po += play_delta;
	}

	po += momentum_delta;

	head->offset = (int)po;

	momentum_delta *= 0.99;
      }

      if (po < startf || po >= endf) break;
    } /* for (i) */

    /* Only write if still playing */
    if (head->going) {

      n = device_write (handle, pbuf, PBUF_SIZE, head->offset);

      g_mutex_lock (s->play_mutex);

      head->realoffset = device_offset (handle);
      if (head->realoffset == -1) {
	head->realoffset = head->offset;
      }

      head->offset = head->realoffset;
      
      if (s->by_user /* && s->play_scrubbing */) {
	head->offset = s->user_offset;
      } else {
	s->user_offset = head->realoffset;
      }

      g_mutex_unlock (s->play_mutex);
    }

  }

  return (head->reverse ? (po <= startf) : (po >= endf));
}
#endif /* play_view: DEVEL_CODE, soon obsolete */

static sw_framecount_t
head_read_unrestricted (sw_head * head, sw_audio_t * buf,
			sw_framecount_t count)
{
  sw_sample * sample = head->sample;
  sw_sounddata * sounddata = sample->sounddata;
  sw_format * f = sounddata->format;
  sw_audio_t * d;
  gdouble po = 0.0, p;
  gfloat relpitch;
  sw_framecount_t i, j, b;
  gint si=0, si_next = 0;
  gboolean interpolate = FALSE;
  gboolean do_smoothing = FALSE;
  sw_framecount_t last_user_offset = -1;
  int pbuf_size = count * f->channels;
  gdouble scrub_rate = f->rate / 30.0;

  d = (sw_audio_t *)sounddata->data;

  b = 0;

  po = head->offset;

  /* compensate for sampling rate of driver */
  relpitch = (gfloat)((gdouble)f->rate / (gdouble)handle->driver_rate);
  
  for (i = 0; i < count; i++) {
    if (head->mute || sample->user_offset == last_user_offset) {
      for (j = 0; j < f->channels; j++) {
	buf[b] = 0.0;
	b++;
      }
    } else {
      si = (int)floor(po);
      
      interpolate = (si < sounddata->nr_frames);
      
      p = po - (gdouble)si;
      si *= f->channels;

      if (interpolate) {
	si_next = si+f->channels;
	for (j = 0; j < f->channels; j++) {
	  buf[b] = head->gain * (d[si] * p + d[si_next] * (1 - p));
	  if (do_smoothing) {
	    sw_framecount_t b1, b2;
	    b1 = (b - f->channels + pbuf_size) % pbuf_size;
	    b2 = (b1 - f->channels + pbuf_size) % pbuf_size;
	    buf[b] += buf[b] * 2.0;
	    buf[b] += buf[b1] * 3.0 + buf[b2] * 4.0;
	    buf[b] /= 10.0;
	  }
	  b++; si++; si_next++;
	}
      } else {
	for (j = 0; j < f->channels; j++) {
	  buf[b] = head->gain * d[si];
	  if (do_smoothing) {
	    sw_framecount_t b1, b2;
	    b1 = (b - f->channels + pbuf_size) % pbuf_size;
	    b2 = (b1 - f->channels + pbuf_size) % pbuf_size;
	    buf[b] += buf[b] * 2.0;
	    buf[b] += buf[b1] * 3.0 + buf[b2] * 4.0;
	    buf[b] /= 10.0;
	  }
	  b++; si++;
	}
      }
    }

    if (head->scrubbing) {
      gfloat new_delta;

      if (sample->by_user) {
	new_delta = (sample->user_offset - po) / scrub_rate;
	
	head->delta = head->delta * 0.9 + new_delta * 0.1;
	
	sample->by_user = FALSE;
	
	last_user_offset = sample->user_offset;
      } else  {
	gfloat new_po, u_po = (gdouble)sample->user_offset;

	new_delta = (sample->user_offset - po) / scrub_rate;
	
	head->delta = head->delta * 0.99 + new_delta * 0.01;
	
	new_po = po + (head->delta * relpitch);
	
	if ((head->delta < 0 && new_po < u_po) ||
	    (head->delta > 0 && new_po > u_po)) {
	  po = u_po;
	  head->delta = 0.0;
	}
      }
      
      do_smoothing = TRUE;

    } else {
      gfloat tdelta = head->rate * sample->rate;
      gfloat hdelta = head->delta * (head->reverse ? -1.0 : 1.0);

      if (sample->by_user) {
	head->offset = (gdouble)sample->user_offset;
	po = head->offset;
	head->delta = tdelta;

	sample->by_user = FALSE;
	do_smoothing = TRUE;

	last_user_offset = sample->user_offset;
      }

      if (hdelta < -0.3 * tdelta || hdelta > 1.001 * tdelta) {
	head->delta *= 0.9999;
      } else if (hdelta < 0.7 * tdelta) {
	head->delta = 0.8 * tdelta * (head->reverse ? -1.0 : 1.0);
      } else if (hdelta < .999 * tdelta) {
	head->delta *= 1.0001;
      } else {
	head->delta = tdelta * (head->reverse ? -1.0 : 1.0);
      }

      do_smoothing = FALSE;
    }

    po += head->delta * relpitch;

    {
      gdouble nr_frames = (gdouble)sample->sounddata->nr_frames;
      if (head->looping) {
	while (po < 0.0) po += nr_frames;
	while (po > nr_frames) po -= nr_frames;
      } else {
	if (po < 0.0) po = 0.0;
	else if (po > nr_frames) po = nr_frames;
      }
    }

    head->offset = po;

  }

  return count;
}

sw_framecount_t
head_read (sw_head * head, sw_audio_t * buf, sw_framecount_t count)
{
  sw_sample * sample = head->sample;
  sw_sounddata * sounddata = sample->sounddata;
  sw_format * f = sounddata->format;
  sw_framecount_t remaining = count, written = 0, n = 0;
  GList * gl;
  sw_sel * sel;

  while (head->going && remaining > 0) {
    n = 0;

    if (head->restricted /* && !head->scrubbing */) {
      g_mutex_lock (sounddata->sels_mutex);

      if (g_list_length (sounddata->sels) == 0) {
	g_mutex_unlock (sounddata->sels_mutex);
	goto zero_pad;
      }

      /* Find selection region that offset is or should be in */
      if (head->reverse) {
	for (gl = g_list_last (sounddata->sels); gl; gl = gl->prev) {
	  sel = (sw_sel *)gl->data;

	  if (head->offset > sel->sel_end)
	    head->offset = sel->sel_end;
	
	  if (head->offset > sel->sel_start) {
	    n = MIN (remaining, head->offset - sel->sel_start);
	    break;
	  }
	}
      } else {
	for (gl = sounddata->sels; gl; gl = gl->next) {
	  sel = (sw_sel *)gl->data;

	  if (head->offset < sel->sel_start)
	    head->offset = sel->sel_start;
	
	  if (head->offset < sel->sel_end) {
	    n = MIN (remaining, sel->sel_end - head->offset);
	    break;
	  }
	}
      }

      g_mutex_unlock (sounddata->sels_mutex);

    } else { /* unrestricted */
      if (head->reverse) {
	n = MIN (remaining, head->offset);
      } else {
	n = MIN (remaining, sounddata->nr_frames - head->offset);
      }
    }

    if (n == 0) {
      if (!head->restricted || sounddata->sels == NULL) {
	head->offset = head->reverse ? sounddata->nr_frames : 0;
      } else {
	g_mutex_lock (sounddata->sels_mutex);
	if (head->reverse) {
	  gl = g_list_last (sounddata->sels);
	  sel = (sw_sel *)gl->data;
	  head->offset = sel->sel_end;
	} else {
	  gl = sounddata->sels;
	  sel = (sw_sel *)gl->data;
	  head->offset = sel->sel_start;
	}
	g_mutex_unlock (sounddata->sels_mutex);
      }

      if (!head->looping) {
	head->going = FALSE;
      }
    } else {
      written += head_read_unrestricted (head, buf, n);
      buf += (int)frames_to_samples (f, n);
      remaining -= n;
    }
  }

 zero_pad:

  if (remaining > 0) {
    n = frames_to_bytes (f, remaining);
    memset (buf, 0, n);
    written += remaining;
  }

  return written;
}

/* initialise a head for playback */
static void
head_init_playback (sw_sample * s)
{
  sw_head * head = s->play_head;
  GList * gl;
  sw_sel * sel;
  sw_framecount_t sels_start, sels_end;

  /*  g_mutex_lock (s->play_mutex);*/

  s->by_user = FALSE;

  if (!head->going) {
    head_set_offset (head, s->user_offset);
    head->delta = head->rate * s->rate;
  }

  if (head->restricted) {
    g_mutex_lock (s->sounddata->sels_mutex);

    gl = s->sounddata->sels;
    sel = (sw_sel *)gl->data;
    sels_start = sel->sel_start;

    gl = g_list_last (s->sounddata->sels);
  
    sel = (sw_sel *)gl->data;
    sels_end = sel->sel_end;
    g_mutex_unlock (s->sounddata->sels_mutex);

    if (head->reverse && head->offset <= sels_start) {
      head_set_offset (head, sels_end);
    } else if (!head->reverse && head->offset >= sels_end) {
      head_set_offset (head, sels_start);
    }
  } else {
    if (head->reverse && head->offset <= 0) {
      head_set_offset (head,  s->sounddata->nr_frames);
    } else if (!head->reverse && head->offset >= s->sounddata->nr_frames) {
      head_set_offset (head, 0);
    }
  }

  /*  g_mutex_unlock (s->play_mutex);*/
}

static void
channel_convert_adding (sw_audio_t * src, int src_channels,
			sw_audio_t * dest, int dest_channels,
			sw_framecount_t n)
{
  int j;
  sw_framecount_t i, b = 0;
  sw_audio_intermediate_t a;

  if (src_channels == 1) { /* mix mono data up */
    for (i = 0; i < n; i++) {
      for (j = 0; j < dest_channels; j++) {
	dest[b] += src[i];
	b++;
      }
    }
  } else if (dest_channels == 1) { /* mix down to mono */
    for (i = 0; i < n; i++) {
      a = 0.0;
      for (j = 0; j < src_channels; j++) {
	a += src[b];
	b++;
      }
      a /= (sw_audio_intermediate_t)src_channels;
      dest[i] += (sw_audio_t)a;
    }
  } else if (src_channels < dest_channels) { /* copy to first channels */
    for (i = 0; i < n; i++) {
      for (j = 0; j < src_channels; j++) {
	dest[i * dest_channels + j] += src[b];
	b++;
      }
    }
  } else if (dest_channels < src_channels) { /* copy first channels only */
    for (i = 0; i < n; i++) {
      for (j = 0; j < dest_channels; j++) {
	dest[b] += src[i * src_channels + j];
	b++;
      }
    }
  } else if (src_channels == dest_channels) { /* just add */
    for (i = 0; i < n * src_channels; i ++) {
      dest[i] += src[i];
    }
  }
}

static void
prepare_play_head (sw_head * head)
{
  sw_sample * s = head->sample;

  g_mutex_lock (play_mutex);

  if (g_list_find (active_play_heads, head) == 0) {
    active_play_heads = g_list_append (active_play_heads, head);
  }

  g_mutex_unlock (play_mutex);

  head_init_playback (s);
}

#ifdef RECORD_DEMO_FILES
static gchar *
generate_demo_filename (void)
{
  return g_strdup_printf ("/tmp/sweep-demo-%d.au", getpid ());
}
#endif

#define PSIZ 64

/* how many inactive writes to do before closing */
#define INACTIVE_TIMEOUT 256

static void
play_head (sw_head * nymhead)
{
  sw_sample * s;
  sw_format * f;
  sw_framecount_t n, count;
  fd_set fds;
  sw_audio_t * pbuf, * devbuf;
  int inactive_writes = 0;
  GList * gl, * gl_next;
  sw_head * head;
  int pbuf_chans;

#ifdef RECORD_DEMO_FILES
  gchar * filename;
  SNDFILE * sndfile;
  SF_INFO sfinfo;
#endif

  if (!(gl = active_play_heads)) return;
  
  head = (sw_head *)gl->data;

  f = head->sample->sounddata->format;

  device_setup (handle, f);

  pbuf = g_malloc (PSIZ * f->channels * sizeof (sw_audio_t));
  pbuf_chans = f->channels;

  devbuf = g_malloc (PSIZ * handle->driver_channels * sizeof (sw_audio_t));

#ifdef RECORD_DEMO_FILES
  filename = generate_demo_filename ();
  sfinfo.samplerate = f->rate;
  sfinfo.channels = handle->driver_channels;
  sfinfo.format = SF_FORMAT_AU | SF_FORMAT_FLOAT | SF_ENDIAN_CPU;
  sndfile = sf_open (filename, SFM_WRITE, &sfinfo);
  if (sndfile == NULL) sf_perror (NULL);
  else printf ("Writing to %s\n", filename);
#endif

  while (!stop_all && inactive_writes < INACTIVE_TIMEOUT) {
    FD_ZERO (&fds);
    FD_SET (handle->driver_fd, &fds);
      
    if (select (handle->driver_fd+1, &fds, NULL, NULL, &tv_instant) == 0);

    n = PSIZ;

    count = n * handle->driver_channels;

    memset (devbuf, 0, PSIZ * handle->driver_channels * sizeof (sw_audio_t));

    if (active_play_heads == NULL) {
      inactive_writes++;
    } else {
      inactive_writes = 0;

      for (gl = active_play_heads; gl; gl = gl_next) {
	head = (sw_head *)gl->data;
	s = head->sample;
	gl_next = gl->next;

	if (!head->going || !sample_bank_contains (head->sample)) {
	  g_mutex_lock (play_mutex);
	  active_play_heads = g_list_remove (active_play_heads, head);
	  g_mutex_unlock (play_mutex);
	} else {

	  f = s->sounddata->format;
	  
	  if (f->channels > pbuf_chans) {
	    pbuf = g_realloc (pbuf, PSIZ * f->channels * sizeof (sw_audio_t));
	    pbuf_chans = f->channels;
	  }
	  
	  head_read (head, pbuf, n);
	  
#if 0
	  if (f->channels != handle->driver_channels) {
	    channel_convert (pbuf, f->channels, devbuf,
			     handle->driver_channels, n);
	  }
#else
	  channel_convert_adding (pbuf, f->channels, devbuf,
				  handle->driver_channels, n);

#endif
	
	  /* XXX: store the head->offset NOW for device_offset referencing */
	  
	  g_mutex_lock (s->play_mutex);
	  
#if 0
	  head->realoffset = head->offset;
#else
	  head->realoffset = device_offset (handle);
	  if (head->realoffset == -1) {
	    head->realoffset = head->offset;
	  }
#endif
	
	  head->offset = head->realoffset;
	  
	  if (s->by_user /* && s->play_scrubbing */) {
	    /*head->offset = s->user_offset;*/
	  } else {
	    if (!head->scrubbing) s->user_offset = head->realoffset;
	  }
	  
	  /*	if (!head->going) active = FALSE;*/
	  
	  g_mutex_unlock (s->play_mutex);
	}
      }
    }
    
    device_write (handle, devbuf, count, -1 /* offset reference */);  

#ifdef RECORD_DEMO_FILES
    if (sndfile) sf_writef_float (sndfile, devbuf,
				  count / handle->driver_channels);
#endif
  }

#if 0
  if (!head->looping) {
    device_drain (handle);

    HEAD_SET_GOING (head, FALSE);
  }
#endif

  device_reset (handle);
  device_close (handle);

#ifdef RECORD_DEMO_FILES
  if (sndfile) {
    printf ("Closing %s\n", filename);
    sf_close (sndfile);
  }
#endif

  player_thread = -1;
}

static gboolean
ensure_playing (void)
{
  sw_handle * h;

  if (player_thread == -1) {
    if ((h = device_open (O_WRONLY)) != NULL) {
      handle = h;
      pthread_create (&player_thread, NULL, (void *) (*play_head), NULL);
      return TRUE;
    } else {
      return FALSE;
    }
  }

  return TRUE;
}

void
sample_play (sw_sample * sample)
{
  sw_head * head = sample->play_head;

  prepare_play_head (head);

  head_set_going (head, TRUE);

  sample_refresh_playmode (sample);

  if (ensure_playing()) {
    start_playmarker (sample);
  } else {
    head_set_going (head, FALSE);
    sample_refresh_playmode (sample);
  }
}

void
play_view_all (sw_view * view)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;

  sample_set_stop_offset (s);
  sample_set_previewing (s, FALSE);
    
  prev_sample = s;

  head_set_restricted (head, FALSE);

  sample_play (s);
#if 0
  prepare_play_head (head);

  head_set_going (head, TRUE);

  sample_refresh_playmode (s);

  if (ensure_playing()) {
    start_playmarker (s);
  } else {
    head_set_going (head, FALSE);
    sample_refresh_playmode (s);
  }
#endif
}

void
play_view_sel (sw_view * view)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;

  if (s->sounddata->sels == NULL) {
    head_set_going (head, FALSE);
    sample_refresh_playmode (s);
    return;
  }

  sample_set_stop_offset (s);
  sample_set_previewing (s, FALSE);

  prev_sample = s;

  head_set_restricted (head, TRUE);

  sample_play (s);

#if 0
  prepare_play_head (head);

  head_set_going (head, TRUE);

  sample_refresh_playmode (s);

  if (ensure_playing()) {
    start_playmarker (s);
  } else {
    head_set_going (head, FALSE);
    sample_refresh_playmode (s);
  }
#endif
}

#ifdef DEVEL_CODE
static void
pvpreview_cursor (sw_view * view)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;
  sw_framecount_t delta, preview_start, preview_end;

  device_setup (handle, s->sounddata->format);

  head_init_playback (s);
  
  delta = time_to_frames (s->sounddata->format, 1.0); /* 1 second */ 

  preview_start = head->offset - delta;
  preview_start = MAX (preview_start, 0);

  preview_end = head->offset + delta;
  preview_end = MIN (preview_end, s->sounddata->nr_frames);

  if (play_view (view, preview_start, preview_end,
		 preview_start, preview_end, 1.0)) {
    g_mutex_lock (head->head_mutex);
    head->offset = head->stop_offset;
    g_mutex_unlock (head->head_mutex);
  }

  device_drain (handle);

  HEAD_SET_GOING (head, FALSE);

  device_reset (handle);
  device_close (handle);
}
#endif

void
play_preview_cursor (sw_view * view)
{
#if 0
  sw_handle * h;

  sw_sample * s = view->sample;

  stop_playback (prev_sample);

  head_set_going (head, TRUE);
  sample_refresh_playmode (s);
  sample_set_stop_offset (s);
  sample_set_previewing (s, TRUE);

  prev_sample = s;

  if ((h = device_open(O_WRONLY)) != NULL) {
    handle = h;
#ifdef DEBUG
    if (player_thread != -1) {
      g_print ("Warning: runaway player thread!\n");
    }
#endif
    pthread_create (&player_thread, NULL, (void *) (*pvpreview_cursor),
		    view);
    start_playmarker (s);
  } else {
    sample_set_playmode (s, SWEEP_TRANSPORT_STOP);
  }
#endif
}

#ifdef DEVEL_CODE
static void
pvpreroll (sw_view * view)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;
  sw_framecount_t delta, preroll_start, preroll_end;

  device_setup (handle, s->sounddata->format);

  head_init_playback (s);

  delta = time_to_frames (s->sounddata->format, 1.0); /* preroll 1 second */
 
  preroll_start = head->reverse ? head->offset : head->offset - delta;
  preroll_start = MAX (preroll_start, 0);

  preroll_end = head->reverse ? head->offset + delta : head->offset;
  preroll_end = MIN (preroll_end, s->sounddata->nr_frames);

  if (play_view (view, preroll_start, preroll_end,
		 preroll_start, preroll_end, 1.0)) {
    g_mutex_lock (head->head_mutex);
    head->offset = head->stop_offset;
    g_mutex_unlock (head->head_mutex);
  }

  device_drain (handle);

  HEAD_SET_GOING (head, FALSE);

  device_reset (handle);
  device_close (handle);
}
#endif

void
play_preroll (sw_view * view)
{
#if 0
  sw_handle * h;

  sw_sample * s = view->sample;

  stop_playback (prev_sample);

  sample_set_playmode (s, SWEEP_TRANSPORT_PLAY);
  sample_set_stop_offset (s);
  sample_set_previewing (s, TRUE);

  prev_sample = s;

  if ((h = device_open(O_WRONLY)) != NULL) {
    handle = h;
#ifdef DEBUG
    if (player_thread != -1) {
      g_print ("Warning: runaway player thread!\n");
    }
#endif
    pthread_create (&player_thread, NULL, (void *) (*pvpreroll), view);
    start_playmarker (s);
  } else {
    sample_set_playmode (s, SWEEP_TRANSPORT_STOP);
  }
#endif
}

#ifdef DEVEL_CODE
static void
pvpreview_cut (sw_view * view)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;
  GList * gl, * gl_prev, * gl_next;
  sw_sel * sel;
  sw_framecount_t delta, preview_start, preview_end;
  sw_framecount_t start = 0, end = 0;
  gboolean was_reverse;
  gboolean skip = FALSE; /* voodoo */

  device_setup (handle, s->sounddata->format);

  head_init_playback (s);

  delta = time_to_frames (s->sounddata->format, 1.0);

  g_mutex_lock (s->sounddata->sels_mutex);

  sel = (sw_sel *)s->sounddata->sels->data;
  preview_start = sel->sel_start;

  for (gl = s->sounddata->sels; gl->next; gl = gl->next);
  
  sel = (sw_sel *)gl->data;
  preview_end = sel->sel_end;
  g_mutex_unlock (s->sounddata->sels_mutex);

  gl = head->reverse ? gl : s->sounddata->sels;
  
  preview_start -= delta;
  preview_start = MAX (preview_start, 0);

  preview_end += delta;
  preview_end = MIN (preview_end, s->sounddata->nr_frames);

  g_mutex_lock (head->head_mutex);
  head->offset = head->reverse ? preview_end : preview_start;
  g_mutex_unlock (head->head_mutex);

  while (gl) {
    
    if (!head->going) break;
    
    /* Hold sels_mutex for as little time as possible */
    g_mutex_lock (s->sounddata->sels_mutex);
    
    /* if gl is no longer in sels, break out */
    if (g_list_position (s->sounddata->sels, gl) == -1) {
      g_mutex_unlock (s->sounddata->sels_mutex);
      break;
    }
    
    sel = (sw_sel *)gl->data;
    start = sel->sel_start;
    end = sel->sel_end;

    gl_prev = gl->prev;
    gl_next = gl->next;

    g_mutex_unlock (s->sounddata->sels_mutex);

    was_reverse = head->reverse;

    if (!skip) {
      if (head->reverse) {
	play_view (view, end, head->offset, end, head->offset, 1.0);
      } else {
	play_view(view, head->offset, start, head->offset, start, 1.0);
      }

      skip = (was_reverse != head->reverse);
    } else {
      skip = FALSE;
    }

    g_mutex_lock (head->head_mutex);

    if (head->reverse) {
      head->offset = start;
      gl = gl_prev;
    } else {
      head->offset = end;
      gl = gl_next;
    }

    g_mutex_unlock (head->head_mutex);

  }

  if (head->going) {
    if (head->reverse) {
      play_view (view, preview_start, start, preview_start, start, 1.0);
    } else {
      play_view (view, end, preview_end, end, preview_end, 1.0);
    }
  }

  g_mutex_lock (head->head_mutex);
  head->offset = head->stop_offset;
  g_mutex_unlock (head->head_mutex);
  
  if (head->going) {
    device_drain (handle);

    HEAD_SET_GOING (head, FALSE);
  }

  device_reset (handle);
  device_close (handle);
}
#endif

void
play_preview_cut (sw_view * view)
{
#if 0
  sw_handle * h;

  sw_sample * s = view->sample;

  if (s->sounddata->sels == NULL) {
    sample_set_playmode (s, SWEEP_TRANSPORT_STOP);
    return;
  }

  stop_playback (prev_sample);

  sample_set_playmode (s, SWEEP_TRANSPORT_PLAY);
  sample_set_stop_offset (s);
  sample_set_previewing (s, TRUE);

  prev_sample = s;

  if ((h = device_open(O_WRONLY)) != NULL) {
    handle = h;
#ifdef DEBUG
    if (player_thread != -1) {
      g_print ("Warning: runaway player thread\n");
    }
#endif
    pthread_create (&player_thread, NULL, (void *) (*pvpreview_cut), view);
    start_playmarker (s);
  } else {
    sample_set_playmode (s, SWEEP_TRANSPORT_STOP);
  }
#endif
}


void
play_view_all_pitch (sw_view * view, gfloat pitch)
{
  sw_sample * s = view->sample;
  sw_head * head = s->play_head;
  sw_framecount_t mouse_offset;

  mouse_offset =
    sample_display_get_mouse_offset (SAMPLE_DISPLAY(view->display));
  sample_set_playmarker (s, mouse_offset, TRUE);

  sample_set_stop_offset (s);
  sample_set_previewing (s, FALSE);

  prev_sample = s;

  head_set_restricted (head, FALSE);

  sample_play (s);
}

void
stop_all_playback (void)
{
  stop_all = TRUE;
  g_list_free (active_play_heads);
  active_play_heads = NULL;
}

void
pause_playback (sw_sample * s)
{
  sw_head * head;

  if (s == NULL) return;

  head = s->play_head;

  if (head->going) {
    head_set_going (head, FALSE);
  }

  sample_set_stop_offset (s);
}

void
stop_playback (sw_sample * s)
{
  sw_head * head;

  if (s == NULL) return;

  head = s->play_head;

  if (head->going) {
    head_set_going (head, FALSE);
    sample_set_playmarker (s, head->stop_offset, TRUE);

    g_mutex_lock (play_mutex);
    active_play_heads = g_list_remove (active_play_heads, head);
    g_mutex_unlock (play_mutex);
  }
}

gboolean
any_playing (void)
{
  return ((player_thread != -1) && (active_play_heads != NULL));
}
