/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 * Copyright (C) 2002 CSIRO Australia
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#ifdef HAVE_LIBSAMPLERATE

#include <stdio.h>
#include <string.h>

#include <samplerate.h>

#include <sweep/sweep_i18n.h>
#include <sweep/sweep_types.h>
#include <sweep/sweep_typeconvert.h>
#include <sweep/sweep_undo.h>
#include <sweep/sweep_filter.h>
#include <sweep/sweep_sounddata.h>
#include <sweep/sweep_sample.h>

#include "sweep_app.h"
#include "edit.h"
#include "sw_chooser.h"

#define BUFFER_LEN 4096

static void
do_samplerate_thread (sw_op_instance * inst)
{
  sw_sample * sample = inst->sample;
  int new_rate = GPOINTER_TO_INT (inst->do_data);
  double src_ratio;

  GList * gl;
  sw_sel * sel;

  sw_format * old_format = sample->sounddata->format;
  sw_sounddata * old_sounddata, * new_sounddata;
  sw_framecount_t old_nr_frames, new_nr_frames;
  int channel = 0;

  sw_audio_t * old_d, * new_d;

  double input[BUFFER_LEN], output[BUFFER_LEN];
  SRC_STATE * src_state;
  SRC_DATA src_data;
  int error;

  sw_framecount_t remaining, offset_in, offset_out, i, run_total, ctotal;
  int percent;

  gboolean active = TRUE;

  if ((src_state = src_init ()) == NULL) {
    /* XXX: error */
  }

  src_ratio = (double)new_rate / (double)old_format->rate;

  src_data.end_of_input = 0;
  src_data.data_in_len = 0;
  src_data.data_in = input;
  src_data.src_ratio = src_ratio;
  src_data.data_out = output;
  src_data.data_out_len = BUFFER_LEN;

  old_sounddata = sample->sounddata;
  old_nr_frames = old_sounddata->nr_frames;

  new_nr_frames = old_nr_frames * src_ratio;
  new_sounddata = sounddata_new_empty (old_format->channels, new_rate,
				       new_nr_frames);

  remaining = old_nr_frames * old_format->channels;
  ctotal = remaining / 100;
  if (ctotal == 0) ctotal = 1;
  run_total = 0;
  channel = 0;
  offset_in = 0, offset_out = 0;


  old_d = (sw_audio_t *)old_sounddata->data;
  new_d = (sw_audio_t *)new_sounddata->data;

  /* Create selections */
  g_mutex_lock (sample->ops_mutex);
  for (gl = old_sounddata->sels; gl; gl = gl->next) {
    sel = (sw_sel *)gl->data;

    sounddata_add_selection_1 (new_sounddata, sel->sel_start * src_ratio,
			       sel->sel_end * src_ratio);
  }
  g_mutex_unlock (sample->ops_mutex);

  /* XXX: move play/rec offsets */

  /* Resample data */
  while (active && remaining > 0) {
    g_mutex_lock (sample->ops_mutex);

    if (sample->edit_state == SWEEP_EDIT_STATE_CANCEL) {
      active = FALSE;
    } else {

      /* if the input buffer is empty, refill it */
      if (src_data.data_in_len == 0) {
	src_data.data_in_len = MIN (old_nr_frames - offset_in, BUFFER_LEN);
	old_d = &((sw_audio_t *)old_sounddata->data)
	  [offset_in * old_format->channels + channel];
	for (i = 0; i < src_data.data_in_len; i++) {
	  input[i] = (double) *old_d;
	  old_d += old_format->channels;
	}
	src_data.data_in = input;

	if (src_data.data_in_len < BUFFER_LEN) {
	  src_data.end_of_input = TRUE;
	}
      }

      if ((error = src_process (src_state, &src_data))) {
	/* XXX: src error */
      }

      new_d = &((sw_audio_t *)new_sounddata->data)
	[offset_out * old_format->channels + channel];
      for (i = 0; i < src_data.data_out_gen; i++) {
	*new_d = (sw_audio_t)output[i];
	new_d += old_format->channels;
      }

      remaining -= (sw_framecount_t)src_data.data_in_used;
      run_total += (sw_framecount_t)src_data.data_in_used;

      /* switch to next channel if done */
      if (src_data.end_of_input && src_data.data_out_gen == 0) {
	channel++;

	offset_in = 0;
	offset_out = 0;

	src_reset (src_state);

	src_data.end_of_input = 0;
	src_data.data_in_len = 0;
	src_data.data_in = input;
	src_data.src_ratio = src_ratio;
      } else {
	offset_in += (sw_framecount_t)src_data.data_in_used;
	offset_out += (sw_framecount_t)src_data.data_out_gen;
      
	src_data.data_in += src_data.data_in_used;
	src_data.data_in_len -= src_data.data_in_used;
      }

      percent = run_total / ctotal;
      sample_set_progress_percent (sample, percent);
    }

    g_mutex_unlock (sample->ops_mutex);
  }

  if (remaining > 0) { /* cancelled or failed */
    sounddata_destroy (new_sounddata);
  } else if (sample->edit_state == SWEEP_EDIT_STATE_BUSY) {
    sample->sounddata = new_sounddata;

    inst->redo_data = inst->undo_data =
      sounddata_replace_data_new (sample, old_sounddata, new_sounddata);

    register_operation (sample, inst);
  }

  /*sample_set_edit_state (sample, SWEEP_EDIT_STATE_DONE);*/
}

static sw_operation samplerate_op = {
  SWEEP_EDIT_MODE_ALLOC,
  (SweepCallback)do_samplerate_thread,
  (SweepFunction)NULL,
  (SweepCallback)undo_by_sounddata_replace,
  (SweepFunction)sounddata_replace_data_destroy,
  (SweepCallback)redo_by_sounddata_replace,
  (SweepFunction)sounddata_replace_data_destroy
};

void
resample (sw_sample * sample, int new_rate)
{
#undef BUF_LEN
#define BUF_LEN 128
  char buf[BUF_LEN];

  g_snprintf (buf, BUF_LEN, _("Resample from %d Hz to %d Hz"),
	      sample->sounddata->format->rate, new_rate);

  schedule_operation (sample, buf, &samplerate_op, GINT_TO_POINTER(new_rate));
}

static void
samplerate_dialog_ok_cb (GtkWidget * widget, gpointer data)
{
  sw_sample * sample = (sw_sample *)data;
  GtkWidget * dialog;
  GtkWidget * chooser;

  int new_rate;

  dialog = gtk_widget_get_toplevel (widget);

  chooser = gtk_object_get_user_data (GTK_OBJECT(dialog));
  new_rate = samplerate_chooser_get_rate (chooser);

  gtk_widget_destroy (dialog);

  resample (sample, new_rate);
}

static void
samplerate_dialog_cancel_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog;

  dialog = gtk_widget_get_toplevel (widget);
  gtk_widget_destroy (dialog);
}


void
samplerate_dialog_new_cb (GtkWidget * widget, gpointer data)
{
  sw_view * view = (sw_view *)data;
  sw_sample * sample = view->sample;
  GtkWidget * dialog;
  GtkWidget * main_vbox;
  GtkWidget * label;
  GtkWidget * chooser;
  GtkWidget * button, * ok_button;

  gchar * current;

  dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW(dialog), _("Sweep: Resample"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_container_set_border_width (GTK_CONTAINER(dialog), 8);

  main_vbox = GTK_DIALOG(dialog)->vbox;

  current = g_strdup_printf (_("Current sample rate: %d Hz"),
			     sample->sounddata->format->rate);
  label = gtk_label_new (current);
  gtk_box_pack_start (GTK_BOX(main_vbox), label, TRUE, TRUE, 8);
  gtk_widget_show (label);

  chooser = samplerate_chooser_new (_("New sampling rate"));
  gtk_box_pack_start (GTK_BOX(main_vbox), chooser, TRUE, TRUE, 0);
  gtk_widget_show (chooser);

  gtk_object_set_user_data (GTK_OBJECT(dialog), chooser);

  /* OK */

  ok_button = gtk_button_new_with_label (_("OK"));
  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (ok_button), GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), ok_button,
		      TRUE, TRUE, 0);
  gtk_widget_show (ok_button);
  gtk_signal_connect (GTK_OBJECT(ok_button), "clicked",
		      GTK_SIGNAL_FUNC (samplerate_dialog_ok_cb),
		      sample);

  /* Cancel */

  button = gtk_button_new_with_label (_("Cancel"));
  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (button), GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button,
		      TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT(button), "clicked",
		      GTK_SIGNAL_FUNC (samplerate_dialog_cancel_cb),
		      NULL);

  gtk_widget_grab_default (ok_button);

  gtk_widget_show (dialog);
}

#endif /* HAVE_LIBSAMPLERATE */
