/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <errno.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <sweep/sweep_i18n.h>
#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>

#include "driver.h"
#include "question_dialogs.h"
#include "preferences.h"

#ifdef DRIVER_OSS

#include <sys/soundcard.h>
#define DEV_DSP "/dev/dsp"

#define LOG_FRAGS 6
#define LOG_FRAGS_MIN 1
#define LOG_FRAGS_MAX 10

#ifdef DEVEL_CODE
/*#define DEBUG*/
#endif

/* #define DEBUG_OFFSET */

typedef struct _oss_play_offset oss_play_offset;

struct _oss_play_offset {
  int framenr;
  sw_framecount_t offset;
};

static oss_play_offset offsets[1<<LOG_FRAGS_MAX];

static gdouble configured_logfrags = LOG_FRAGS;
static int nfrags = 0;

#define LOGFRAGS_TO_FRAGS(l) (1 << ((int)(floor((l)) - 1)))

static GtkWidget * dialog = NULL;

static char * configured_devicename = DEV_DSP;

static int oindex;
static int current_frame;
static int frame;

static GtkWidget * combo;
static GtkObject * adj;

#define DEV_KEY "OSS_Device"
#define LOG_FRAGS_KEY "OSS_Logfrags"

static void
config_dev_dsp_dialog_ok_cb (GtkWidget * widget, gpointer data)
{
  GtkAdjustment * adj = GTK_ADJUSTMENT(data);
  GtkWidget * dialog;

  /*g_print ("value: %d\n", (int)floor(adj->value));*/
  configured_logfrags = adj->value;

  prefs_set_int (LOG_FRAGS_KEY, configured_logfrags);

  if (configured_devicename != DEV_DSP)
    g_free (configured_devicename);

  configured_devicename =
    g_strdup(gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(combo)->entry)));

  prefs_set_string (DEV_KEY, configured_devicename);

  dialog = gtk_widget_get_toplevel (widget);
  gtk_widget_hide (dialog);
}

static void
config_dev_dsp_dialog_cancel_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog;

  dialog = gtk_widget_get_toplevel (widget);
  gtk_widget_hide (dialog);
}

static void
default_devicename_cb (GtkWidget * widget, gpointer data)
{
  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(combo)->entry), DEV_DSP);
}

static void
config_dev_dsp (void)
{
  GtkWidget * frame;
  GtkWidget * hbox;
  GtkWidget * vbox;
  GtkWidget * label;
  GtkWidget * hscale;
  GtkWidget * ok_button;
  GtkWidget * button;

  GList * cbitems = NULL;

  int * clf;
  char * cdev;

  if (dialog == NULL) {

    clf = prefs_get_int (LOG_FRAGS_KEY);
    if (clf != NULL) {
      configured_logfrags = (gdouble) *clf;
      free (clf);
    }

    cdev = prefs_get_string (DEV_KEY);
    if (cdev != NULL) {
      configured_devicename = cdev;
      /* XXX: free previous configured_device ?? */
    }
    
    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW(dialog), "Sweep: OSS audio device configuration");
    gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
    gtk_container_border_width (GTK_CONTAINER(dialog), 8);

    /* Device name */

    hbox = gtk_hbox_new (FALSE, 8);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 8);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Device:"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    cbitems = NULL;
    cbitems = g_list_append (cbitems, "/dev/dsp");
    cbitems = g_list_append (cbitems, "/dev/dsp1");
    cbitems = g_list_append (cbitems, "/dev/sound/dsp");

    combo = gtk_combo_new ();
    gtk_box_pack_start (GTK_BOX(hbox), combo, TRUE, TRUE, 0);
    gtk_widget_show (combo);

    gtk_combo_set_popdown_strings (GTK_COMBO(combo), cbitems);

    button = gtk_button_new_with_label (_("Default"));
    gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(default_devicename_cb), NULL);

    /* Buffering */

    frame = gtk_frame_new (_("Device buffering"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), frame,
			TRUE, TRUE, 8);
    gtk_widget_show (frame);

    vbox = gtk_vbox_new (FALSE, 8);
    gtk_container_add (GTK_CONTAINER(frame), vbox);
    gtk_widget_show (vbox);

    label = gtk_label_new (_("Varying this slider controls the lag between "
			     "cursor movements and playback. This is "
			     "particularly noticeable when \"scrubbing\" "
			     "during playback.\n\nLower values improve "
			     "responsiveness but may degrade audio quality "
			     "on heavily-loaded systems."));
    gtk_label_set_line_wrap (GTK_LABEL(label), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    hbox = gtk_hbox_new (FALSE, 8);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 8);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Low latency /\nMore dropouts"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 8);
    gtk_widget_show (label);

    adj = gtk_adjustment_new (configured_logfrags, /* value */
			      LOG_FRAGS_MIN, /* lower */
			      LOG_FRAGS_MAX+1, /* upper */
			      1, /* step incr */
			      1, /* page incr */
			      1  /* page size */
			      );

    hscale = gtk_hscale_new (GTK_ADJUSTMENT(adj));
    gtk_box_pack_start (GTK_BOX(hbox), hscale, TRUE, TRUE, 4);
    gtk_scale_set_draw_value (GTK_SCALE(hscale), TRUE);
    gtk_range_set_update_policy (GTK_RANGE(hscale), GTK_UPDATE_CONTINUOUS);
    gtk_widget_set_usize (hscale, 160, -1);
    gtk_widget_show (hscale);

    label = gtk_label_new (_("High latency /\nFewer dropouts"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 8);
    gtk_widget_show (label);

    /* Changes ... info */
    label = gtk_label_new (_("Changes to device settings will take effect on"
			     " next playback."));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), label,
			FALSE, FALSE, 0);
    gtk_widget_show (label);

    /* OK */

    ok_button = gtk_button_new_with_label (_("OK"));
    GTK_WIDGET_SET_FLAGS (GTK_WIDGET (ok_button), GTK_CAN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), ok_button,
			TRUE, TRUE, 0);
    gtk_widget_show (ok_button);
    gtk_signal_connect (GTK_OBJECT(ok_button), "clicked",
			GTK_SIGNAL_FUNC (config_dev_dsp_dialog_ok_cb),
			adj);

    /* Cancel */

    button = gtk_button_new_with_label (_("Cancel"));
    GTK_WIDGET_SET_FLAGS (GTK_WIDGET (button), GTK_CAN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button,
			TRUE, TRUE, 0);
    gtk_widget_show (button);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC (config_dev_dsp_dialog_cancel_cb),
			NULL);

    gtk_widget_grab_default (ok_button);
  }

  if (configured_devicename == NULL) {
    configured_devicename = DEV_DSP;
  }
  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(combo)->entry),
		      configured_devicename);

  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), configured_logfrags);

  if (!GTK_WIDGET_VISIBLE(dialog)) {
    gtk_widget_show(dialog);
  } else {
    gdk_window_raise(dialog->window);
  }
}


/* driver functions */

static sw_handle *
open_dev_dsp (int flags)
{
  int dev_dsp;
  sw_handle * handle;
  int i;

  if (configured_devicename == NULL) {
    configured_devicename = DEV_DSP;
  }

  flags &= O_RDONLY|O_WRONLY|O_RDWR; /* mask out direction flags */

  if((dev_dsp = open(configured_devicename, flags, 0)) == -1) {
    sweep_perror (errno, "Unable to open device %s", configured_devicename);
    return NULL;
  }

  handle = g_malloc0 (sizeof (sw_handle));
  handle->driver_fd = dev_dsp;

  oindex = 0;
  current_frame = 0;
  for (i = 0; i < (LOGFRAGS_TO_FRAGS(LOG_FRAGS_MAX)); i++) {
    offsets[i].framenr = 0;
    offsets[i].offset = -1;
  }
  frame = 0;

  return handle;
}

static void
setup_dev_dsp (sw_handle * handle, sw_format * format)
{
  int dev_dsp;
  /*  int mask, format, stereo, frequency;*/

  int stereo = 0;
  int bits;
  int i, want_channels, channels;
  int srate;
  int error;
  int fragsize, frag;
  int fmt;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in setup()\n");
#endif
    return;
  }

  dev_dsp = handle->driver_fd;

  if (ioctl (dev_dsp, SNDCTL_DSP_STEREO, &stereo) == -1) {
    /* Fatal error */
    perror("open_dsp_device 2 ") ;
    exit (1);
  } ;

  if (ioctl (dev_dsp, SNDCTL_DSP_RESET, 0)) {
    perror ("open_dsp_device 3 ") ;
    exit (1) ;
  } ;

  nfrags = LOGFRAGS_TO_FRAGS(configured_logfrags);
  fragsize = 8;
  frag = (nfrags << 16) | fragsize;
  if ((error = ioctl (dev_dsp, SNDCTL_DSP_SETFRAGMENT, &frag)) != 0) {
    perror ("OSS: error setting fragments");
  }

  fragsize = (frag & 0xffff);
  nfrags = (frag & 0x7fff000)>>16;
#ifdef DEBUG
  g_print ("Got %d frags of size 2^%d\n", nfrags, fragsize);
#endif
    
  bits = 16 ;
  if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_BITS, &bits)) != 0) {
    perror ("open_dsp_device 4 ");
    exit (1);
  }

  for (i=1; i <= format->channels; i *= 2) {
    channels = format->channels / i;
    want_channels = channels;

    if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_CHANNELS, &channels)) == -1) {
      perror ("open_dsp_device 5 ") ;
      exit (1) ;
    }

    if (channels == want_channels) break;
  }

  handle->driver_channels = channels;

  srate = format->rate;

  if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_RATE, &srate)) != 0) {
    perror ("open_dsp_device 6 ") ;
    exit (1) ;
  }

  handle->driver_rate = srate;

  if ((error = ioctl (dev_dsp, SNDCTL_DSP_SYNC, 0)) != 0) {
    perror ("open_dsp_device 7 ") ;
    exit (1) ;
  }

  fmt = AFMT_QUERY;
  if ((error = ioctl (dev_dsp, SOUND_PCM_SETFMT, &fmt)) != 0) {
    perror ("open_dsp_device 8") ;
    exit (1) ;
  }

  handle->custom_data = GINT_TO_POINTER(0);

#ifdef WORDS_BIGENDIAN
  if (fmt == AFMT_S16_LE || fmt == AFMT_U16_LE) {
    handle->custom_data = GINT_TO_POINTER(1);
  }
#else
  if (fmt == AFMT_S16_BE || fmt == AFMT_U16_BE) {
    handle->custom_data = GINT_TO_POINTER(1);
  }
#endif

#ifdef DEBUG
  {
    int caps;

    if (ioctl (dev_dsp, SNDCTL_DSP_GETCAPS, &caps) == -1) {
      sweep_perror (errno, "OSS: Unable to get device capabilities");
    }
    /* CAP_REALTIME tells whether or not this device can give exact
     * DMA pointers via GETOSPACE/GETISPACE. If this is true, then
     * the device reports with byte accuracy. If it is false it reports
     * to at least the nearest fragment bound, which is still pretty
     * good for small fragments, so it's not much of a problem if
     * this capability is not present.
     */
    g_print ("Realtime: %s\n", caps & DSP_CAP_REALTIME ? "YES" : "NO");
  }
#endif
}

#define RECORD_SCALE (SW_AUDIO_T_MAX / 32768.0)

static ssize_t
read_dev_dsp (sw_handle * handle, sw_audio_t * buf, size_t count)
{
  gint16 * bbuf;
  size_t byte_count;
  ssize_t bytes_read;
  int need_bswap;
  int i;

  byte_count = count * sizeof (gint16);
  bbuf = alloca (byte_count);
  bytes_read = read (handle->driver_fd, bbuf, byte_count);

  if (bytes_read == -1) {
    sweep_perror (errno, "Error reading from OSS audio device");
    return -1;
  }

  need_bswap = GPOINTER_TO_INT(handle->custom_data);

  if (need_bswap) {
    unsigned char * ucptr = (unsigned char *)bbuf;
    unsigned char temp;
    
    for (i = 0; i < count; i++) {
      temp = ucptr[2 * i];
      ucptr[2 * i] = ucptr [2 * i + 1];
      ucptr[2 * i + 1] = temp;
    }
  }

  for (i = 0; i < count; i++) {
    buf[i] = (sw_audio_t)(bbuf[i] * RECORD_SCALE);
  }

  return (bytes_read / sizeof (gint16));
}

#define PLAYBACK_SCALE (32768 / SW_AUDIO_T_MAX)

static ssize_t
write_dev_dsp (sw_handle * handle, sw_audio_t * buf, size_t count,
	       sw_framecount_t play_offset)
{
  gint16 * bbuf;
  size_t byte_count;
  ssize_t bytes_written;
  int need_bswap;
  int i;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in write()\n");
#endif
    return -1;
  }

  current_frame += count;
  offsets[oindex].framenr = current_frame;
  offsets[oindex].offset = play_offset;
  oindex++; oindex %= nfrags;

  byte_count = count * sizeof (gint16);
  bbuf = alloca (byte_count);

  for (i = 0; i < count; i++) {
    bbuf[i] = (gint16)(PLAYBACK_SCALE * buf[i]);
  }

  need_bswap = GPOINTER_TO_INT(handle->custom_data);

  if (need_bswap) {
    unsigned char * ucptr = (unsigned char *)bbuf;
    unsigned char temp;
    
    for (i = 0; i < count; i++) {
      temp = ucptr[2 * i];
      ucptr[2 * i] = ucptr [2 * i + 1];
      ucptr[2 * i + 1] = temp;
    }
  }

  bytes_written = write (handle->driver_fd, bbuf, byte_count);

  if (bytes_written == -1) {
    sweep_perror (errno, "Error writing to OSS audio device");
    return -1;
  } else {
    return (bytes_written / sizeof(gint16));
  }
}

static sw_framecount_t
offset_dev_dsp (sw_handle * handle)
{
  count_info info;
  int i, o;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in offset()\n");
#endif
    return -1;
  }

  if (ioctl (handle->driver_fd, SNDCTL_DSP_GETOPTR, &info) == -1) {
#ifdef DEBUG_OFFSET
    g_print ("error in GETOPTR\n");
#endif
    return -1;
  }

  frame = info.bytes;
#ifdef DEBUG_OFFSET
  g_print ("frame: %d\n", frame);
#endif

  o = oindex+1;
  for (i = 0; i < nfrags; i++) {
    o %= nfrags;
#ifdef DEBUG_OFFSET
    g_print ("\t(%d) Checking %d: %d\n", frame, o, offsets[o].framenr);
#endif
    if (offsets[o].framenr >= frame) {
      return offsets[o].offset;
    }
    o++;
  }

  return -1;
}

static void
reset_dev_dsp (sw_handle * handle)
{
  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in reset()\n");
#endif
    return;
  }

  if(ioctl (handle->driver_fd, SNDCTL_DSP_RESET, 0) == -1) {
    sweep_perror (errno, "Error resetting OSS audio device");
  }
}

static void
flush_dev_dsp (sw_handle * handle)
{
}

void
drain_dev_dsp (sw_handle * handle)
{
  if (handle == NULL) {
    g_print ("handle NULL in drain ()\n");
    return;
  }

  if(ioctl (handle->driver_fd, SNDCTL_DSP_POST, 0) == -1) {
    sweep_perror (errno, "POST error on OSS audio device");
  }

  if (ioctl (handle->driver_fd, SNDCTL_DSP_SYNC, 0) == -1) {
    sweep_perror (errno, "SYNC error on OSS audio device");
  }
}

static void
close_dev_dsp (sw_handle * handle)
{
  close (handle->driver_fd);
}

static sw_driver _driver_oss = {
  config_dev_dsp,
  open_dev_dsp,
  setup_dev_dsp,
  read_dev_dsp,
  write_dev_dsp,
  offset_dev_dsp,
  reset_dev_dsp,
  flush_dev_dsp,
  drain_dev_dsp,
  close_dev_dsp,
};

#else

static sw_driver _driver_oss = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
};

#endif

sw_driver * driver_oss = &_driver_oss;
